function [nverts,fnverts] = convert2fiji(varargin)
%This little function "con" vert matlab x, y, z, image cooridinates to FIJI
%x, y, z, coordinates.
%Synatax:   [nvert,fnvert] = convert2fiji('vert',verts,'cf',[1 1 0.7],'rnd',200,'sav',1);
%Input:
%   verts = the matlab vertices dataset, note it will be assumed that the
%           it's an array of at least 3 columns and the first three are x,
%           y, and z.  If no verts are entered, the user will be prompted
%           to open vert files.
%   cf = the conversion factor for the x,y,z positions, which is used if
%       any of those positions have been scaled.  Default = [1 1 1]
%   rnd = if you want the output to be a random subset of the original.
%           by default, rndout = 0, which is all out, to specify the number
%           of random vertices just replace 0 by any integer, e.g., 300
%           vertices.
%   sav = save the verts out as a file or not.  Default = 0.
%   fiji = if turned off will not convert the verts to fiji, useful when
%       you just want this function to modify the x,y,z scaling.  Default =
%       1
%Output:
%   nverts = the FIJI vertices dataset, it will only be a three column
%           array.  If the entered dataset is more than three columns it
%           will be truncated.
%   fnverts = the non-truncated version, so if the entering array is only
%           three columns this will be the same as nvert.

[verts,cf,rndout,sav,filenames,pathname,fiji] = parse(varargin);

if ~iscell(verts)    %make the vertices dataset a cell array
    verts = {verts};
end

for i = 1:size(verts,2);
    %pluck vert out into a temp variable
    vert = verts{1,i};
    if fiji
        vtmp = vert(:,1:2)-1;   %shift by 1 pixel from 1,1 origin to 0,0 origin
        vtmp = [vtmp(:,2)./cf(1) vtmp(:,1)./cf(2) vert(:,3)./cf(3)];    %swap around the x and y columns, and scale if needed
        vtmp = round(vtmp);
    else    %just scale
        vtmp = [vert(:,2)./cf(1) vert(:,1)./cf(2) vert(:,3)./cf(3)];
    end
    
    if rndout   %the user desires a subset
        rndidx = randi(size(vert,1),rndout,1);     %generate the random index
        %lets make sure th random index is unique
        rndidx = unique(rndidx);
        while size(rndidx,1)<rndout    %not all verts are unique
            rndtmp = randi(size(vert,1),rndout,1);  %generate again
            rndidx = vertcat(rndidx,rndtmp);    %combine
            rndidx = unique(rndidx);        %unique out
            if size(rndidx,1)>rndout    %if more than needed
                rndidx = rndidx(1:rndout,:);   %truncate
            end
        end
        nvert = vtmp(rndidx,:);     %select the subset
        %nvert = round(nvert);   %round off any non-integers
        nprop = vert(rndidx,:);     %grab the properties for the dataset
    else
        nvert = vtmp;
        %nvert = round(vtmp);   %round off any non-integers
        nprop = vert;       %take it all
    end
    
    if size(vert,2)>3       %there are extraneous data
        fnvert = horzcat(nvert,nprop);      %tack on the other stuff
    else
        fnvert = nvert;     %the same
    end
    
    if sav  %if you want to save this data out
        if isempty(filenames)   %the vertices are entered not opened
            sav2csv(nvert);
        else
            if fiji
                sav2csv(nvert,['FIJIvert_',filenames{i},'_',num2str(size(nvert,1))],pathname);    %save the vertices
            else
                sav2csv(nvert,['revert_',filenames{i},'_',num2str(size(nvert,1))],pathname);    %save the vertices
            end
        end
    end
    nverts{i} = nvert;     %output
    fnverts{i} = fnvert;    %output
end


%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [verts,cf,rndout,sav,filenames,pathname,fiji] = parse(input)

verts = [];
cf = [1 1 1];
rndout = 0;
sav = 0;
filenames = [];
pathname = [];
fiji = 1;

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'vert'
                    verts = input{1,i+1};
                case 'cf'
                    cf = input{1,i+1};
                case 'rnd'
                    rndout = input{1,i+1};
                case 'sav'
                    sav = input{1,i+1};
                case 'fiji'
                    fiji = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end

if isempty(verts) %no vertices entered lets open some files then
    [filenames,pathname,filterindex] = uigetfile2({'*.csv','Text files (*.csv)';'*.*','All Files';},...
        'Open vertices files','Multiselect','on');
    if ischar(filenames)    %convert to cell
        filenames = {filenames};
    end
    for i = 1:size(filenames,2)
        verts{i} = single(dlmread([pathname,filenames{1,i}],',',1,0));
    end
end